import time
import subprocess
import os
import sys

# 設定ファイルを読み込む
try:
    import settings
except ImportError:
    print("エラー: settings.py が見つかりません。")
    sys.exit(1)

# パスの設定
WATCH_FILE = settings.WATCH_FILE_PATH
SCRIPTS_DIR = settings.SCRIPTS_DIR

# スキル（キーワードと実行するスクリプト）のマッピング
SKILL_MAPPING = {
    "AIニュース": "ai_news_researcher.py",
    "ブログ作成": "AGENT_REQUEST", # Antigravityが直接担当するスキル
}

def safe_write(file_path, content_lines, retries=5, delay=1.0):
    """iCloud同期などの競合を防ぐため、リトライ付きで安全に書き込む"""
    temp_path = file_path + ".tmp"
    for i in range(retries):
        try:
            # 1. 一時ファイルに書き込む
            with open(temp_path, 'w', encoding='utf-8') as f:
                f.writelines(content_lines)
            
            # 2. 一時ファイルを本番ファイルに置き換える（Atomic Write）
            if os.path.exists(file_path):
                os.replace(temp_path, file_path)
            else:
                os.rename(temp_path, file_path)
            return True
        except Exception as e:
            print(f"書き込みリトライ中 ({i+1}/{retries}): {e}")
            time.sleep(delay)
    return False

def check_for_instructions():
    """Obsidianのノートを監視し、指示があれば実行する"""
    if not os.path.exists(WATCH_FILE):
        print(f"待機中... (ファイルが見つかりません: {WATCH_FILE})")
        return

    with open(WATCH_FILE, 'r', encoding='utf-8') as f:
        lines = f.readlines()

    new_lines = []
    task_executed = False

    for line in lines:
        # 未完了のタスク ("- [ ] 指示:") を探す
        if "- [ ] 指示:" in line:
            full_instruction = line.split("指示:")[1].strip()
            print(f"指示を受信: {full_instruction}")

            # 引数の抽出（「」や""で囲まれた部分）
            argument = ""
            if "「" in full_instruction and "」" in full_instruction:
                argument = full_instruction.split("「")[1].split("」")[0]
            elif '"' in full_instruction:
                 parts = full_instruction.split('"')
                 if len(parts) > 1:
                     argument = parts[1]
            
            # キーワードに基づいてスキルを実行
            script_to_run = None
            for keyword, script in SKILL_MAPPING.items():
                if keyword in full_instruction:
                    script_to_run = script
                    break
            
            if script_to_run == "AGENT_REQUEST":
                print(f"エージェント（Antigravity）への依頼を検知: {instruction}")
                # 行を更新 ("- [?]" に変更し、ステータスを追記)
                new_line = line.replace("- [ ]", "- [?]").strip() + " (Antigravityに依頼中)\n"
                new_lines.append(new_line)
                task_executed = True

            elif script_to_run:
                print(f"スキルを実行中: {script_to_run}...")
                
                script_path = os.path.join(SCRIPTS_DIR, script_to_run)
                
                try:
                    # サブプロセスとしてスクリプトを実行 (引数がある場合は渡す)
                    cmd = ['python3', script_path]
                    if argument:
                        cmd.append(argument)
                    
                    result = subprocess.run(cmd, capture_output=True, text=True)
                    
                    if result.returncode == 0:
                        status = "完了"
                        detail = "成功しました"
                        print(f"実行成功: {result.stdout}")
                    else:
                        status = "エラー"
                        detail = f"失敗しました: {result.stderr}"
                        print(f"実行失敗: {result.stderr}")

                except Exception as e:
                    status = "エラー"
                    detail = f"予期せぬエラー: {str(e)}"
                    print(f"システムエラー: {e}")

                # 行を更新 ("- [x]" に変更し、結果を追記)
                timestamp = time.strftime("%Y-%m-%d %H:%M:%S")
                new_line = line.replace("- [ ]", "- [x]").strip() + f" ({status}: {timestamp})\n"
                new_lines.append(new_line)
                new_lines.append(f"\t- 結果: {detail}\n")
                task_executed = True
            
            else:
                # 対応するスキルが見つからない場合
                print("対応するスキルが見つかりませんでした。")
                new_line = line.replace("- [ ]", "- [?]").strip() + " (未対応の指示)\n"
                new_lines.append(new_line)
                task_executed = True

        else:
            new_lines.append(line)

    # ファイルを更新（変更があった場合のみ）
    if task_executed:
        if safe_write(WATCH_FILE, new_lines):
            print("タスクリストを正常に更新しました。")
        else:
            print("エラー: タスクリストの更新に失敗しました。")

def main():
    print("=== AIコマンドセンター 監視を開始しました ===")
    print(f"監視対象: {WATCH_FILE}")
    print("終了するには Ctrl+C を押してください")
    
    try:
        while True:
            check_for_instructions()
            time.sleep(5) # 5秒ごとにチェック
    except KeyboardInterrupt:
        print("\n監視を終了します。")

if __name__ == "__main__":
    main()
