import time
import subprocess
import os
import sys

# 設定ファイルを読み込む
try:
    import settings
except ImportError:
    print("エラー: settings.py が見つかりません。")
    sys.exit(1)

# パスの設定
WATCH_FILE = settings.WATCH_FILE_PATH
SCRIPTS_DIR = settings.SCRIPTS_DIR

# スキル（キーワードと実行するスクリプト）のマッピング
SKILL_MAPPING = {
    "AIニュース": "ai_news_researcher.py",
    # 新しいスキルを追加する場合はここに追記します
}

def check_for_instructions():
    """Obsidianのノートを監視し、指示があれば実行する"""
    if not os.path.exists(WATCH_FILE):
        print(f"待機中... (ファイルが見つかりません: {WATCH_FILE})")
        return

    with open(WATCH_FILE, 'r', encoding='utf-8') as f:
        lines = f.readlines()

    new_lines = []
    task_executed = False

    for line in lines:
        # 未完了のタスク ("- [ ] 指示:") を探す
        if "- [ ] 指示:" in line:
            instruction = line.split("指示:")[1].strip()
            print(f"指示を受信: {instruction}")

            # キーワードに基づいてスキルを実行
            script_to_run = None
            for keyword, script in SKILL_MAPPING.items():
                if keyword in instruction:
                    script_to_run = script
                    break
            
            if script_to_run:
                print(f"スキルを実行中: {script_to_run}...")
                
                script_path = os.path.join(SCRIPTS_DIR, script_to_run)
                
                try:
                    # サブプロセスとしてスクリプトを実行
                    result = subprocess.run(['python3', script_path], capture_output=True, text=True)
                    
                    if result.returncode == 0:
                        status = "完了"
                        detail = "成功しました"
                        print(f"実行成功: {result.stdout}")
                    else:
                        status = "エラー"
                        detail = f"失敗しました: {result.stderr}"
                        print(f"実行失敗: {result.stderr}")

                except Exception as e:
                    status = "エラー"
                    detail = f"予期せぬエラー: {str(e)}"
                    print(f"システムエラー: {e}")

                # 行を更新 ("- [x]" に変更し、結果を追記)
                timestamp = time.strftime("%Y-%m-%d %H:%M:%S")
                new_line = line.replace("- [ ]", "- [x]").strip() + f" ({status}: {timestamp})\n"
                new_lines.append(new_line)
                new_lines.append(f"\t- 結果: {detail}\n")
                task_executed = True
            
            else:
                # 対応するスキルが見つからない場合
                print("対応するスキルが見つかりませんでした。")
                new_line = line.replace("- [ ]", "- [?]").strip() + " (未対応の指示)\n"
                new_lines.append(new_line)
                task_executed = True

        else:
            new_lines.append(line)

    # ファイルを更新（変更があった場合のみ）
    if task_executed:
        with open(WATCH_FILE, 'w', encoding='utf-8') as f:
            f.writelines(new_lines)
        print("タスクリストを更新しました。")

def main():
    print("=== AIコマンドセンター 監視を開始しました ===")
    print(f"監視対象: {WATCH_FILE}")
    print("終了するには Ctrl+C を押してください")
    
    try:
        while True:
            check_for_instructions()
            time.sleep(5) # 5秒ごとにチェック
    except KeyboardInterrupt:
        print("\n監視を終了します。")

if __name__ == "__main__":
    main()
